/*
 * Copyright (c) 2002-2007 TeamDev Ltd. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * The complete licence text can be found at
 * http://www.teamdev.com/winpack/license.jsf
 */
package teamdev.jxcapture.samples.demo;

import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.Application;

import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;

/**
 * @author Ikryanov Vladimir
 */
public class ImageComponent extends JPanel {

    private int[] percents;

    private int defaultIndex;
    private int currentIndex;

    private double percentWidth;
    private double percentHeight;

    private BufferedImage originImage;
    private Image fastImage;

    private ImagePanel imagePanel = new ImagePanel();

    public ImageComponent() {
        ApplicationContext context = Application.getInstance().getContext();
        ResourceMap resource = context.getResourceMap(ImageComponent.class);

        setLayout(new BorderLayout());
        add(imagePanel, BorderLayout.CENTER);
        setName("imageComponent");

        currentIndex = defaultIndex = resource.getInteger("defaultIndex");
        percents = new int[resource.getInteger("percents.legth")];
        for (int i = 0; i < percents.length; i++) {
            percents[i] = resource.getInteger("percents[" + i + "]");
        }

        resource.injectComponents(this);
    }

    public void setImage(File file) throws Exception {
        BufferedImage image = ImageIO.read(file);
        setImage(image);
    }

    public void setImage(BufferedImage image) {
        initialize(image);
        fastImage = image.getScaledInstance(image.getWidth(), image.getHeight(), BufferedImage.SCALE_FAST);
        imagePanel.updateImage(fastImage);
    }

    public void saveImage(File file) throws Exception {
        String fileName = file.getName();
        String extension = fileName.substring(fileName.lastIndexOf(".") + 1);
        ImageIO.write(originImage, extension, file);
    }

    public void zoomIn() {
        if (canZoomIn()) zoomTo(++currentIndex);
    }

    public void zoomOut() {
        if (canZoomOut()) zoomTo(--currentIndex);
    }

    public boolean canZoomIn() {
        return !(currentIndex == percents.length - 1);
    }

    public boolean canZoomOut() {
        return !(currentIndex == 0);
    }

    public int[] getPercents() {
        return percents;
    }

    public void actualSize() {
        imagePanel.updateImage(fastImage);
        currentIndex = defaultIndex;
    }

    public void fitToWindow(Dimension maxSize) {
        int imageWidth = originImage.getWidth();
        int imageHeight = originImage.getHeight();
        Dimension sourceSize = new Dimension(imageWidth, imageHeight);
        Dimension scaledSize = getScaledDimension(sourceSize, maxSize);
        Image scaledImage = originImage.getScaledInstance(
                scaledSize.width, scaledSize.height, Image.SCALE_FAST);
        imagePanel.updateImage(scaledImage);
    }

    public void dispose() {
        imagePanel.dispose();
        imagePanel = null;

        originImage.flush();
        originImage = null;
    }

    public void zoomTo(int index) {
        Image scaledImage = originImage;
        currentIndex = index;
        if (currentIndex != defaultIndex) {
            int width = (int) (percentWidth * percents[index]);
            int height = (int) (percentHeight * percents[index]);
            scaledImage = originImage.getScaledInstance(width, height, Image.SCALE_FAST);
        }
        imagePanel.updateImage(scaledImage);
    }

    public BufferedImage getImage() {
        return originImage;
    }

    private static Dimension getScaledDimension(Dimension sourceSize, Dimension maxSize) {
        int width;
        int height;

        double sourceAspectRatio = (double) sourceSize.width / (double) sourceSize.height;
        double destAspectRatio = (double) maxSize.width / (double) maxSize.height;

        if (sourceAspectRatio > destAspectRatio) {
            width = maxSize.width;
            height = (int) (width / sourceAspectRatio);
        } else {
            height = maxSize.height;
            width = (int) (height * sourceAspectRatio);
        }

        if (width == 0 && maxSize.width > 0)  width = 1;
        if (height == 0 && maxSize.height > 0) height = 1;

        return new Dimension(width, height);
    }

    private void initialize(BufferedImage image) {
        percentWidth = (double) image.getWidth() / 100;
        percentHeight = (double) image.getHeight() / 100;
        originImage = image;
    }

    public class ImagePanel extends JLabel {

        private ImageIcon imageIcon;

        public ImagePanel() {
            setHorizontalAlignment(CENTER);
            setVerticalAlignment(CENTER);
        }

        public void updateImage(Image image) {
            if (imageIcon != null) {
                dispose();
            }
            imageIcon = new ImageIcon(image);
            setIcon(imageIcon);
        }

        public void dispose() {
            if (imageIcon != null) {
                imageIcon.getImage().flush();
                imageIcon = null;
            }
        }
    }
}
